/*******************************************************************************
 * Copyright (c) 2022, 2023 THALES GLOBAL SERVICES. All rights reserved.
 *
 * Contributors: Obeo - initial API and implementation
 *******************************************************************************/
package fr.obeo.dsl.viewpoint.collab.common.internal.security;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.KeySpec;
import java.util.Arrays;
import java.util.Base64;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.PBEKeySpec;
import javax.crypto.spec.SecretKeySpec;

import org.eclipse.equinox.internal.security.storage.SecurePreferencesMapper;
import org.eclipse.equinox.internal.security.storage.SecurePreferencesWrapper;
import org.eclipse.equinox.security.storage.ISecurePreferences;
import org.eclipse.equinox.security.storage.SecurePreferencesFactory;
import org.eclipse.equinox.security.storage.StorageException;

import fr.obeo.dsl.viewpoint.collab.common.internal.Activator;

/**
 * A service class to read credentials from the secure storage to send http request.
 * 
 * @author lfasani
 *
 */
public class HttpClientSecureStorageService {

    private static final String PWD_ATT = "password"; //$NON-NLS-1$

    private static final String LOGIN_ATT = "login"; //$NON-NLS-1$

    private static final String SEPARATOR = ":"; //$NON-NLS-1$

    private static final String ENCRYPTION_ALGORYTHM = "AES/CBC/PKCS5Padding"; //$NON-NLS-1$

    private static final String ENCRYPTION_KEY_ALGORITHM = "AES"; //$NON-NLS-1$

    private static final String PBKDF_ALGORITHM = "PBKDF2WithHmacSHA256"; //$NON-NLS-1$

    private static final char[] CHAR_ARRAY = { 'U', 'n', 's', 'w', 'o', 'r', 'n', '6', 'F', 'a', 'l', 'c', 'o', 'n', '4', 'S', 'h' };

    private ISecurePreferences preferences;

    /**
     * Default constructor.
     */
    public HttpClientSecureStorageService() {
        this.preferences = SecurePreferencesFactory.getDefault();
    }

    /**
     * Default constructor.
     * 
     * @param preferences
     *            The {@link ISecurePreferences}. if null, the default on is used.
     * @param httpPort
     *            used to create the node in the secure storage that will be used by the application client
     */
    public HttpClientSecureStorageService(ISecurePreferences preferences) {
        if (preferences != null) {
            this.preferences = preferences;
        } else {
            this.preferences = SecurePreferencesFactory.getDefault();
        }
    }

    /**
     * Close the secure storage.
     */
    public void close() {
        SecurePreferencesMapper.close(((SecurePreferencesWrapper) this.preferences).getContainer().getRootData());
        SecurePreferencesMapper.clearDefault();
    }

    /**
     * Gets the credentials available in the secure storage.
     * 
     * @param hostName
     *            hostName of the admin server
     * @param httpPort
     *            port of the admin server
     * @param httpLogin
     *            the user for which the password is searched. If null, the couple user/password for the server is
     *            returned.
     */
    public HttpCredentials loadCredentialsFromSecureStorage(String httpHostName, int httpPort, String httpLogin) {
        ISecurePreferences credentialsNode = preferences.node(getToolsCredentialsSecureStorageKey());

        if (credentialsNode != null) {
            // @formatter:off
            Optional<ISecurePreferences> serverNode = Arrays.asList(credentialsNode.childrenNames()).stream()
                    .filter(name -> name.equals(httpHostName + SEPARATOR + httpPort))
                    .findFirst()
                    .map(name-> credentialsNode.node(name));
            // @formatter:on
            if (serverNode.isPresent()) {
                try {
                    String userAttr = serverNode.get().get(LOGIN_ATT, null);
                    boolean validUser = userAttr != null;
                    if (httpLogin != null && !httpLogin.isEmpty()) {
                        validUser = validUser && httpLogin.equals(userAttr);
                    }
                    if (validUser) {
                        String tokenAttr = serverNode.get().get(PWD_ATT, null);
                        Optional<String> decryptedToken = decrypt(tokenAttr);
                        if (decryptedToken.isPresent()) {
                            return new HttpCredentials(userAttr, decryptedToken.get().toCharArray());
                        }
                    }
                } catch (StorageException e) {
                    Activator.getDefault().getLog().error("Impossible to retrieve the user credentials from the secure storage for the server: " + httpHostName + SEPARATOR + httpPort, e); //$NON-NLS-1$
                }
            }
        }
        Activator.getDefault().getLog().error("Impossible to retrieve the user credentials from the secure storage for the server: " + httpHostName + SEPARATOR + httpPort); //$NON-NLS-1$
        return null;
    }

    /**
     * Sets the credentials in the secure storage.
     * 
     * @param hostName
     *            hostName of the admin server
     * @param httpPort
     *            port of the admin server
     * @param httpLogin
     *            the user for which the password is searched. If null, the couple user/password for the server is
     *            returned.
     * @return true if the information has been properly saved in the secure storage.
     */
    public boolean setCredentialsIntoSecureStorage(String httpHostName, int httpPort, String httpLogin, char[] password) throws StorageException, IOException {
        ISecurePreferences credentialsNode = preferences.node(getToolsCredentialsSecureStorageKey());
        ISecurePreferences serverNode = credentialsNode.node(httpHostName + SEPARATOR + httpPort);

        Optional<String> encryptedPassword = this.encrypt(String.valueOf(password));
        if (encryptedPassword.isPresent()) {
            serverNode.put(LOGIN_ATT, httpLogin, false);
            serverNode.put(PWD_ATT, encryptedPassword.get(), false);
            credentialsNode.flush();
            return true;
        }
        return false;
    }

    /**
     * Return whether the secure-storage has been initialized with the tools http credentials.
     * 
     * @return true if the secure-storage contains the expected root node, false otherwise.
     */
    public boolean isHttpToolsCredentialsInitialized(String httpHostName, int httpPort) {
        boolean nodeExists = false;
        if (preferences.nodeExists(getToolsCredentialsSecureStorageKey())) {
            ISecurePreferences credentialsNode = preferences.node(getToolsCredentialsSecureStorageKey());
            String serverNodeName = httpHostName + SEPARATOR + httpPort;
            nodeExists = credentialsNode.nodeExists(serverNodeName);
        }
        return nodeExists;
    }

    /**
     * Gets the password from any admin http tools credentials available in the secure storage.
     */
    public Optional<String> getAnyAdminHttpToolsPwdFromSecureStorage() {
        Optional<String> adminPwd = Optional.empty();
        ISecurePreferences credentialsNode = preferences.node(getToolsCredentialsSecureStorageKey());

        if (credentialsNode != null) {
            adminPwd = Arrays.asList(credentialsNode.childrenNames()).stream()//
                    .map(name -> credentialsNode.node(name))//
                    .filter(node -> {
                        try {
                            return "admin".equals(node.get(LOGIN_ATT, null)); //$NON-NLS-1$
                        } catch (StorageException e) {
                            Activator.getDefault().getLog().error("Impossible to read the node " + node.name(), e); //$NON-NLS-1$
                        }
                        return false;
                    })//
                    .findFirst()//
                    .map(node -> {
                        try {
                            String tokenAttr = node.get(PWD_ATT, null);
                            String decryptedToken = decrypt(tokenAttr).orElse(null);
                            return decryptedToken;
                        } catch (StorageException e) {
                            Activator.getDefault().getLog().error("Impossible to decrypt the password of the node " + node.name(), e); //$NON-NLS-1$
                        }
                        return null;
                    }).filter(Objects::nonNull);
        }
        return adminPwd;
    }

    /**
     * Clear the credentials in the secure storage.
     * 
     * @param hostName
     *            hostName of the admin server
     * @param httpPort
     *            port of the admin server
     * @return true if the credentials has been properly removed from the secure storage.
     */
    public boolean clearCredentialsIntoSecureStorage(String httpHostName, int httpPort) {
        if (preferences.nodeExists(getToolsCredentialsSecureStorageKey())) {
            ISecurePreferences credentialsNode = preferences.node(getToolsCredentialsSecureStorageKey());
            String serverNodeName = httpHostName + SEPARATOR + httpPort;
            if (credentialsNode.nodeExists(serverNodeName)) {
                credentialsNode.node(serverNodeName).removeNode();
                try {
                    preferences.flush();
                    return true;
                } catch (IOException e) {
                    Activator.getDefault().getLog().error("Impossible to clear the credentials.", e); //$NON-NLS-1$
                }
            }
        }
        return false;
    }

    private String getToolsCredentialsSecureStorageKey() {
        return SecureStorageKeys.HTTP_CREDENTIALS;
    }

    private Optional<String> encrypt(String input) {
        Optional<SecretKey> key = getCipherKey();
        if (key.isPresent()) {
            try {
                Cipher cipher = getCipher(Cipher.ENCRYPT_MODE, key.get());
                String salt = UUID.randomUUID().toString().substring(0, 16);
                byte[] cipherText = cipher.doFinal((salt + input).getBytes(StandardCharsets.UTF_8));
                String encodedString = Base64.getEncoder().encodeToString(cipherText);

                return Optional.of(encodedString);
            } catch (InvalidKeyException | InvalidAlgorithmParameterException | NoSuchAlgorithmException | NoSuchPaddingException | IllegalBlockSizeException | BadPaddingException e) {
                Activator.getDefault().getLog().error("Password encryption error."); //$NON-NLS-1$
            }
        }
        return Optional.empty();
    }

    private Optional<String> decrypt(String cipherText) {
        Optional<SecretKey> key = getCipherKey();
        if (key.isPresent()) {
            try {
                Cipher cipher = getCipher(Cipher.DECRYPT_MODE, key.get());
                byte[] plainText = cipher.doFinal(Base64.getDecoder().decode(cipherText));
                return Optional.of(new String(plainText).substring(16));
            } catch (InvalidKeyException | InvalidAlgorithmParameterException | NoSuchAlgorithmException | NoSuchPaddingException | IllegalBlockSizeException | BadPaddingException e) {
                Activator.getDefault().getLog().error("Password decryption error."); //$NON-NLS-1$
            }
        }
        return Optional.empty();
    }

    private Cipher getCipher(int opmode, SecretKey key) throws NoSuchAlgorithmException, NoSuchPaddingException, InvalidKeyException, InvalidAlgorithmParameterException {
        IvParameterSpec ivParameterSpec = this.generateInitializationVector();
        Cipher cipher = Cipher.getInstance(ENCRYPTION_ALGORYTHM);
        cipher.init(opmode, key, ivParameterSpec);
        return cipher;
    }

    private IvParameterSpec generateInitializationVector() {
        byte[] iv = "RU!^Mg29r45GJ5t!".getBytes(StandardCharsets.UTF_8); //$NON-NLS-1$
        return new IvParameterSpec(iv);
    }

    private Optional<SecretKey> getCipherKey() {
        Optional<SecretKey> secret = Optional.empty();
        SecretKeyFactory factory;
        try {
            factory = SecretKeyFactory.getInstance(PBKDF_ALGORITHM);
            KeySpec spec = new PBEKeySpec(CHAR_ARRAY, "277a94da".getBytes(StandardCharsets.UTF_8), 1000, 256); //$NON-NLS-1$
            secret = Optional.of(new SecretKeySpec(factory.generateSecret(spec).getEncoded(), ENCRYPTION_KEY_ALGORITHM));
        } catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
            Activator.getDefault().getLog().error("Password encryption/decryption initialization error."); //$NON-NLS-1$
        }
        return secret;
    }

}
