/*******************************************************************************
 * Copyright (c) 2022, 2023 THALES GLOBAL SERVICES. All rights reserved.
 *
 * Contributors: Obeo - initial API and implementation
 *******************************************************************************/
package fr.obeo.dsl.viewpoint.collab.common.internal.security;

import java.io.IOException;
import java.security.NoSuchAlgorithmException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.eclipse.equinox.internal.security.storage.SecurePreferencesMapper;
import org.eclipse.equinox.internal.security.storage.SecurePreferencesWrapper;
import org.eclipse.equinox.security.storage.ISecurePreferences;
import org.eclipse.equinox.security.storage.SecurePreferencesFactory;
import org.eclipse.equinox.security.storage.StorageException;

import fr.obeo.dsl.viewpoint.collab.common.internal.Activator;

/**
 * A service class to read credentials from the secure storage.
 * 
 * @author fbarbin
 *
 */
@SuppressWarnings("restriction")
public class SecureStorageService {

    private static final String ADMIN_ATT = "isAdmin"; //$NON-NLS-1$

    private static final String USER_TOKENS_NODE = "userTokens"; //$NON-NLS-1$

    private static final String DEFAULT_TOKEN_ID = "default"; //$NON-NLS-1$

    private static final String DEFAULT_USER_ADMIN = "admin"; //$NON-NLS-1$

    private static final String SERVER_HOST_NAME_FROM_CLIENT_APPLICATION = "localhost"; //$NON-NLS-1$

    private ISecurePreferences preferences;

    private TokenManager tokenManager;

    private int httpPort;

    /**
     * Default constructor.
     * 
     * @param preferences
     *            The {@link ISecurePreferences}. if null, the default on is used.
     * @param httpPort
     *            used to create the node in the secure storage that will be used by the application client
     */
    public SecureStorageService(ISecurePreferences preferences, TokenManager tokenManager, int httpPort) {
        if (preferences != null) {
            this.preferences = preferences;
        } else {
            this.preferences = SecurePreferencesFactory.getDefault();
        }
        this.tokenManager = tokenManager;
        this.httpPort = httpPort;
    }

    /**
     * Constructor that uses default ISecurePreferences.
     */
    public SecureStorageService(TokenManager tokenManager) {
        this(null, tokenManager, -1);
    }

    /**
     * Loads credentials available in the secure storage.
     * 
     * @return
     * @throws StorageException
     */
    public List<SecureStorageAdminAuthenticationInfo> loadCredentialsFromSecureStorage() throws StorageException {
        List<SecureStorageAdminAuthenticationInfo> authenticationInfos = new ArrayList<>();
        String securePreferencesNodeName = getHttpServerCredentialsPreferenceNodeName();
        ISecurePreferences node = preferences.node(securePreferencesNodeName);
        for (String subNodes : node.childrenNames()) {
            getAuthenticationInfoFromUserNode(node.node(subNodes)).ifPresent(authenticationInfos::add);
        }
        return authenticationInfos;
    }

    /**
     * Close the secure storage.
     */
    public void close() {
        SecurePreferencesMapper.close(((SecurePreferencesWrapper) this.preferences).getContainer().getRootData());
        SecurePreferencesMapper.clearDefault();
    }

    private Optional<SecureStorageAdminAuthenticationInfo> getAuthenticationInfoFromUserNode(ISecurePreferences userNode) throws StorageException {
        String userId = userNode.name();
        if (userId == null) {
            return Optional.empty();
        }
        boolean isAdmin = userNode.getBoolean(ADMIN_ATT, false);
        ISecurePreferences tokensNode = userNode.node(USER_TOKENS_NODE);
        Map<String, char[]> tokensMap = getTokenFromNode(userNode, tokensNode);
        return Optional.of(new SecureStorageAdminAuthenticationInfo(userId, isAdmin, tokensMap));
    }

    private Map<String, char[]> getTokenFromNode(ISecurePreferences userNode, ISecurePreferences tokensNode) throws StorageException {
        Map<String, char[]> tokensMap = new HashMap<>();
        for (String tokenId : tokensNode.keys()) {
            String tokenValue = tokensNode.get(tokenId, null);
            char[] tokenAsChar = tokenValue.toCharArray();
            if (tokenValue != null) {
                tokensMap.put(tokenId, tokenAsChar);
            }
        }
        return tokensMap;
    }

    /**
     * Generate the tools http credentials in the secure-storage for the right httpPort, if not yet initialized .
     */
    public void generateHttpToolsCredentialsIfNecessary() {
        HttpClientSecureStorageService httpClientSecureStorageService = new HttpClientSecureStorageService(preferences);

        if (!httpClientSecureStorageService.isHttpToolsCredentialsInitialized(SERVER_HOST_NAME_FROM_CLIENT_APPLICATION, httpPort)) {
            Optional<String> anyAdminHttpToolsPwdFromSecureStorage = httpClientSecureStorageService.getAnyAdminHttpToolsPwdFromSecureStorage();
            if (anyAdminHttpToolsPwdFromSecureStorage.isPresent()) {
                try {
                    httpClientSecureStorageService.setCredentialsIntoSecureStorage(SERVER_HOST_NAME_FROM_CLIENT_APPLICATION, httpPort, DEFAULT_USER_ADMIN,
                            anyAdminHttpToolsPwdFromSecureStorage.get().toCharArray());
                } catch (IOException | StorageException e) {
                    Activator.getDefault().getLog().error(MessageFormat.format("Impossible to generate http tools credentials for httpPort \"{0}\"", httpPort), e); //$NON-NLS-1$
                }
            }
        }
    }

    /**
     * Generates the initial admin user with the returned token.
     * 
     * @return the new generated token or null if the initial default admin user already exist.
     * @throws StorageException
     *             if something went wrong with the secure storage.
     * @throws IOException
     *             If the Secure-storage state cannot be persisted.
     */
    public Optional<char[]> generateInitialAdminToken() throws StorageException, IOException {
        if (!isHttpServerCredentialsInitialized()) {
            ISecurePreferences rootNode = getRootNode();
            Optional<char[]> passwordOpt = createUser(DEFAULT_USER_ADMIN, rootNode, true);
            if (passwordOpt.isPresent()) {
                new HttpClientSecureStorageService(preferences).setCredentialsIntoSecureStorage(SERVER_HOST_NAME_FROM_CLIENT_APPLICATION, httpPort, DEFAULT_USER_ADMIN, passwordOpt.get());
            }
            return passwordOpt;
        }
        return Optional.empty();
    }

    private Optional<char[]> createUser(String userID, ISecurePreferences rootNode, boolean isAdmin) throws StorageException, IOException {
        String userNodeID = userID.trim();
        if (!rootNode.nodeExists(userNodeID)) {
            ISecurePreferences userNode = rootNode.node(userNodeID);
            userNode.putBoolean(ADMIN_ATT, isAdmin, false);
            Optional<char[]> optionalToken = addOrReplaceToken(userNode, DEFAULT_TOKEN_ID);
            if (optionalToken.isEmpty()) {
                userNode.removeNode();
            } else {
                rootNode.flush();
                return optionalToken;
            }
        }
        return Optional.empty();
    }

    private Optional<char[]> addOrReplaceToken(ISecurePreferences userNode, String tokenId) throws StorageException, IOException {
        ISecurePreferences tokensNode = userNode.node(USER_TOKENS_NODE);
        char[] token = this.tokenManager.generate();
        Optional<char[]> encrypted = this.tokenManager.encrypt(token);
        if (encrypted.isPresent()) {
            tokensNode.put(tokenId, String.valueOf(encrypted.get()), false);
            userNode.flush();
            return Optional.of(token);
        }
        return Optional.empty();
    }

    /**
     * Checks if provided credentials are valid.
     * 
     * @param user
     *            the user name.
     * @param token
     *            the token.
     * @param requireAdmin
     *            if the user is supposed to have admin privilege.
     * @return true if the credential are valid, false otherwise.
     */
    public boolean isValidCredentials(String user, char[] token, boolean requireAdmin) {
        if (isHttpServerCredentialsInitialized()) {
            try {
                List<SecureStorageAdminAuthenticationInfo> secureStorage = loadCredentialsFromSecureStorage();
                for (SecureStorageAdminAuthenticationInfo authenticationInfo : secureStorage) {
                    if (user.equals(authenticationInfo.getUserID())) {
                        return checkUserCrendentials(token, requireAdmin, authenticationInfo);
                    }
                }
            } catch (StorageException | NoSuchAlgorithmException e) {
                Activator.getDefault().getLog().error("Impossible to verify credentials", e); //$NON-NLS-1$
            }
        }
        return false;

    }

    private boolean checkUserCrendentials(char[] token, boolean requireAdmin, SecureStorageAdminAuthenticationInfo authenticationInfo) throws NoSuchAlgorithmException {
        if (!requireAdmin || authenticationInfo.isAdmin()) {
            Map<String, char[]> tokens = authenticationInfo.getTokens();
            Optional<char[]> encrypted = tokenManager.encrypt(token);

            return encrypted.filter(encryptedToken -> {
                for (char[] currentToken : tokens.values()) {
                    if (Arrays.equals(encryptedToken, currentToken)) {
                        return true;
                    }
                }
                return false;
            }).isPresent();
        }
        return false;
    }

    /**
     * Revokes the given token for the given userId.
     * 
     * @param userId
     *            the userId owning the token.
     * @param tokenId
     *            the token ID to revoke.
     * @return true if the token has been removed or false if it can't be:
     *         <ul>
     *         <li>The "default" token can't be removed</li>
     *         <li>The token does not exist</li>
     *         <li>An exception occurred</li>
     *         </ul>
     */
    public boolean revokeToken(String userId, String tokenId) {
        if (!DEFAULT_TOKEN_ID.equals(tokenId)) {
            ISecurePreferences userNode = getUserNode(userId);
            if (userNode != null) {
                ISecurePreferences tokensNode = userNode.node(USER_TOKENS_NODE);
                try {
                    if (tokensNode.get(tokenId, null) != null) {
                        tokensNode.remove(tokenId);
                        return true;
                    }
                } catch (StorageException e) {
                    Activator.getDefault().getLog().error("Impossible to revoke the token " + tokenId, e); //$NON-NLS-1$
                }
            }
        }
        return false;
    }

    private ISecurePreferences getUserNode(String userId) {
        if (isHttpServerCredentialsInitialized()) {
            ISecurePreferences rootNode = getRootNode();
            String userNodeID = userId;
            if (rootNode.nodeExists(userNodeID)) {
                return rootNode.node(userNodeID);
            }
        }
        return null;
    }

    private ISecurePreferences getRootNode() {
        ISecurePreferences rootNode = preferences.node(getHttpServerCredentialsPreferenceNodeName());
        return rootNode;
    }

    /**
     * Creates a new user in the secure storage.
     * 
     * @param userName
     *            the user name.
     * @param isAdmin
     *            if the user should have admin privilege.
     * @return the default generated token or empty if the user can't be created.
     */
    public Optional<char[]> createNewUser(String userName, boolean isAdmin) {
        if (isHttpServerCredentialsInitialized()) {
            ISecurePreferences rootNode = getRootNode();
            try {
                return createUser(userName, rootNode, isAdmin);
            } catch (StorageException | IOException e) {
                Activator.getDefault().getLog().error("Impossible to create the user " + userName, e); //$NON-NLS-1$
            }
        }
        return Optional.empty();
    }

    private String getHttpServerCredentialsPreferenceNodeName() {
        return SecureStorageKeys.SERVER_REST_ADMIN;
    }

    /**
     * Return whether the secure-storage has been initialized.
     * 
     * @return true if the secure-storage contains the expected root node, false otherwise.
     */
    public boolean isHttpServerCredentialsInitialized() {
        String securePreferencesNodeName = getHttpServerCredentialsPreferenceNodeName();
        return preferences.nodeExists(securePreferencesNodeName);
    }

    /**
     * Clears existing credentials from the secure-storage.
     * 
     * @throws IOException
     *             if something went wrong while flushing the changes.
     */
    public void clearSecureStorageCredentials() throws IOException {
        if (isHttpServerCredentialsInitialized()) {
            ISecurePreferences node = getRootNode();
            node.removeNode();
            this.preferences.flush();
        }

    }

    /**
     * Deletes the given user from the secure storage.
     * 
     * @param userId
     *            the user to delete.
     * @throws IOException
     *             If something went wrong with the secure storage.
     * @return true if the user has been deleted. False otherwise: the user does not exist, the secure storage is not
     *         initialized, an exception occurred.
     */
    public boolean deleteUser(String userId) throws IOException {
        if (isHttpServerCredentialsInitialized()) {
            ISecurePreferences rootNode = getRootNode();
            String userNodeID = userId;
            if (rootNode.nodeExists(userNodeID)) {
                ISecurePreferences userNode = rootNode.node(userNodeID);
                userNode.removeNode();
                rootNode.flush();
                return true;
            }
        }
        return false;
    }

    /**
     * Generates a new token for the given user and token Ids. Creates the token if it does not exist or replaces it
     * otherwise.
     * 
     * @param userId
     *            the user id.
     * @param tokenId
     *            the token id.
     * @return the token if it as been properly generated and saved in the secure storage. Empty otherwise.
     */
    public Optional<char[]> generateToken(String userId, String tokenId) {
        ISecurePreferences userNode = getUserNode(userId);
        if (userNode != null) {
            try {
                return addOrReplaceToken(userNode, tokenId);
            } catch (StorageException | IOException e) {
                Activator.getDefault().getLog().error(MessageFormat.format("Impossible to create or replace the token \"{0}\"", tokenId), e); //$NON-NLS-1$
            }
        }
        return Optional.empty();
    }

    /**
     * Provides the registered user list.
     * 
     * @return A list of User ID.
     */
    public List<String> getUsers() {
        List<String> users = new ArrayList<>();
        ISecurePreferences rootNode = getRootNode();
        for (String childName : rootNode.childrenNames()) {
            users.add(childName);
        }
        return users;
    }

    /**
     * Provides the given user token list.
     * 
     * @param userId
     *            the user id to retrieve tokens.
     * @return the list of tokens.
     */
    public List<String> getTokens(String userId) {
        List<String> tokens = new ArrayList<>();
        try {
            List<SecureStorageAdminAuthenticationInfo> secureStorage = loadCredentialsFromSecureStorage();
            for (SecureStorageAdminAuthenticationInfo authenticationInfo : secureStorage) {
                if (userId.equals(authenticationInfo.getUserID())) {
                    tokens.addAll(authenticationInfo.getTokens().keySet());
                }
            }
        } catch (StorageException e) {
            Activator.getDefault().getLog().error(MessageFormat.format("Impossible to retrieve tokens list for \"{0}\"", userId), e); //$NON-NLS-1$
        }
        return tokens;
    }

}
