/*******************************************************************************
 * Copyright (c) 2021, 2025 THALES GLOBAL SERVICES All rights reserved.
 * 
 * Contributors: Obeo - initial API and implementation
 *******************************************************************************/
package fr.obeo.dsl.viewpoint.collab.common.internal.http.helper;

import java.io.File;
import java.io.IOException;
import java.net.Authenticator;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.net.http.HttpResponse.BodyHandlers;
import java.util.Collections;
import java.util.Map;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.jetty.util.StringUtil;
import org.eclipse.jetty.util.resource.PathResource;
import org.eclipse.jetty.util.ssl.SslContextFactory;

import fr.obeo.dsl.viewpoint.collab.common.internal.Activator;
import fr.obeo.dsl.viewpoint.collab.common.internal.helper.CollabConfigurationHelper;

/**
 * Helper to call http REST service.
 * 
 * @author lfasani
 */
public final class HttpRequestHelper {
    private static final String HTTPS = "https"; //$NON-NLS-1$

    private HttpRequestHelper() {
    }

    /**
     * Call an http request.
     * 
     * @param fullUrl
     *            the URL to request
     * @param requestProperties
     *            the request properties to set in the request header
     * @param response
     *            the response body
     * @return true if the http return code is a success<br>
     *         false otherwise
     */
    public static boolean callHttpService(String fullUrl, Map<String, String> requestProperties, StringBuilder response) {
        boolean result = false;

        try {
            java.net.URI requestURI = new java.net.URI(fullUrl);

            HttpResponse<String> httpResponse = sendHttpRequest(requestURI, requestProperties);
            response.append(httpResponse.body());
            result = httpResponse.statusCode() <= 299;
        } catch (URISyntaxException e) {
            Activator.getDefault().log(IStatus.ERROR, e.getMessage(), e);
        }

        return result;
    }

    /**
     * Send the given Http URI as an {@link HttpRequest} and process the result as an {@link HttpResponse}.
     * 
     * @param requestURI
     *            the URI to send as an {@link HttpRequest}
     * @param authenticator
     *            the {@link Authenticator} to use with the {@link HttpRequest}
     * @param requestProperties
     *            the request properties to set in the request header
     * @return The HttpResponse returned by sending the Http request.
     */
    public static HttpResponse<String> sendHttpRequest(URI requestURI, Map<String, String> requestProperties) {
        return sendHttpRequest(requestURI, null, requestProperties);
    }

    /**
     * Send the given Http URI as an {@link HttpRequest} and process the result as an {@link HttpResponse}.
     * 
     * @param requestURI
     *            the URI to send as an {@link HttpRequest}
     * @param authenticator
     *            the {@link Authenticator} to use with the {@link HttpRequest}
     * @return The HttpResponse returned by sending the Http request.
     */
    public static HttpResponse<String> sendHttpRequest(URI requestURI, Authenticator authenticator) {
        return sendHttpRequest(requestURI, authenticator, Collections.emptyMap());
    }

    /**
     * Send the given Http URI as an {@link HttpRequest} and process the result as an {@link HttpResponse}.
     * 
     * @param requestURI
     *            the URI to send as an {@link HttpRequest}
     * @param authenticator
     *            the {@link Authenticator} to use with the {@link HttpRequest}
     * @param requestProperties
     *            the request properties
     * @return The HttpResponse returned by sending the Http request.
     */
    public static HttpResponse<String> sendHttpRequest(URI requestURI, Authenticator authenticator, Map<String, String> requestProperties) {

        HttpRequest.Builder httpRequestBuilder = HttpRequest.newBuilder() //
                .uri(requestURI) //
                .GET();

        requestProperties.forEach((k, v) -> {
            httpRequestBuilder.header(k, v);
        });
        HttpRequest httpRequest = httpRequestBuilder.build();

        return sendHttpRequest(httpRequest, authenticator);
    }

    /**
     * Send the given Http URI as an {@link HttpRequest} and process the result as an {@link HttpResponse}.
     * 
     * @param httpRequest
     *            an {@link HttpRequest}
     * @param authenticator
     *            the {@link Authenticator} to use with the {@link HttpRequest}
     * @param requestProperties
     *            the request properties
     * @return The HttpResponse returned by sending the Http request.
     */
    public static HttpResponse<String> sendHttpRequest(HttpRequest httpRequest, Authenticator authenticator) {
        HttpResponse<String> response = null;
        URI requestURI = httpRequest.uri();
        try {

            HttpClient.Builder clientBuilder = HttpClient.newBuilder();

            if (authenticator != null) {
                clientBuilder.authenticator(authenticator);
            }

            if (HTTPS.equalsIgnoreCase(requestURI.getScheme())) { // $NON-NLS-1$
                SslContextFactory.Client sslContextFactory = HttpRequestHelper.createSslContextFactory();
                try {
                    sslContextFactory.start();
                    // CHECKSTYLE:OFF
                } catch (Exception e) {
                    // CHECKSTYLE:ON
                    Activator.getDefault().log(IStatus.ERROR, "Error while creating the SSLContext for the HTTPRequest '" + requestURI + "'", e); //$NON-NLS-1$ //$NON-NLS-2$
                    return null;
                }
                clientBuilder.sslContext(sslContextFactory.getSslContext());
            }

            response = clientBuilder.build().send(httpRequest, BodyHandlers.ofString());
        } catch (IOException | InterruptedException e) {
            Activator.getDefault().log(IStatus.ERROR, "Error while sending the HTTPRequest '" + requestURI + "'", e); //$NON-NLS-1$ //$NON-NLS-2$
        }
        return response;
    }

    // Copied and adapted from org.eclipse.net4j.internal.wss.WSSClientConnector
    // See also fr.obeo.dsl.viewpoint.collab.CDOSiriusActivator which configure WSSClientConnector
    private static SslContextFactory.Client createSslContextFactory() {
        // initialize SSL Context
        SslContextFactory.Client sslContextFactory = new SslContextFactory.Client();

        boolean trustAll = Boolean.getBoolean("fr.obeo.dsl.viewpoint.collab.https.jetty.ssl.context.trustall"); //$NON-NLS-1$
        String eia = System.getProperty("fr.obeo.dsl.viewpoint.collab.https.jetty.ssl.context.endpointIdentificationAlgorithm"); //$NON-NLS-1$
        String passphrase = CollabConfigurationHelper.getProperty("fr.obeo.dsl.viewpoint.collab.https.jetty.ssl.context.passphrase", null); //$NON-NLS-1$
        String trusturi = System.getProperty("fr.obeo.dsl.viewpoint.collab.https.jetty.ssl.context.trust"); //$NON-NLS-1$
        String trustType = System.getProperty("fr.obeo.dsl.viewpoint.collab.https.jetty.ssl.context.trust.type"); //$NON-NLS-1$
        String trustAlg = System.getProperty("fr.obeo.dsl.viewpoint.collab.https.jetty.ssl.context.trust.manager.factory.algorithm"); //$NON-NLS-1$

        sslContextFactory.setTrustAll(trustAll);

        if ("null".equals(eia)) { //$NON-NLS-1$
            sslContextFactory.setEndpointIdentificationAlgorithm(null);
        } else if (!StringUtil.isEmpty(eia)) {
            sslContextFactory.setEndpointIdentificationAlgorithm(eia);
        }

        if (trusturi != null) {
            File file = new File(URI.create(trusturi));
            if (file.exists()) {
                sslContextFactory.setTrustStoreResource(new PathResource(file));
                sslContextFactory.setTrustStorePassword(passphrase);
                if (!StringUtil.isEmpty(trustType)) {
                    sslContextFactory.setTrustStoreType(trustType);
                }

                if (!StringUtil.isEmpty(trustAlg)) {
                    sslContextFactory.setTrustManagerFactoryAlgorithm(trustAlg);
                }
            }
        }
        return sslContextFactory;
    }
}
