/*******************************************************************************
 * Copyright (c) 2025, 2026 THALES GLOBAL SERVICES.
 * All rights reserved.
 *
 * Contributors:
 *    Obeo - initial API and implementation
 *******************************************************************************/
package fr.obeo.dsl.viewpoint.collab.common.internal.jwt;

import java.nio.charset.StandardCharsets;
import java.security.Key;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.eclipse.jetty.util.StringUtil;

import com.google.gson.Gson;

import fr.obeo.dsl.viewpoint.collab.common.internal.http.helper.HttpRequestHelper;
import fr.obeo.dsl.viewpoint.collab.common.internal.openid.OpenIdConnectConstants;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Header;
import io.jsonwebtoken.Identifiable;
import io.jsonwebtoken.IncorrectClaimException;
import io.jsonwebtoken.Jws;
import io.jsonwebtoken.JwsHeader;
import io.jsonwebtoken.Jwt;
import io.jsonwebtoken.JwtException;
import io.jsonwebtoken.JwtParser;
import io.jsonwebtoken.JwtParserBuilder;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.security.Jwk;
import io.jsonwebtoken.security.Jwks;
import io.jsonwebtoken.security.SecurityException;
import io.jsonwebtoken.security.SignatureException;

/**
 * Helper to validate received OpenID Connect received id_token and optional access_token for the current OpenID
 * authentication based on the Implicit Flow.
 * 
 * This helper uses the JJWT library.
 * 
 * References:
 * <ul>
 * <li>JJWT: https://github.com/jwtk/jjwt</li>
 * <li>OpenID Connect Core 1.0: https://openid.net/specs/openid-connect-core-1_0.html
 * <li>Other OpenID Connect specs: https://openid.net/developers/specs/</li>
 * </ul>
 * 
 * @author mporhel
 */
public abstract class AbstractJWTHelper {

    /**
     * Nonce claime name.
     */
    protected static final String NONCE_CLAIM = "nonce"; //$NON-NLS-1$

    /**
     * Default user info.
     */
    protected static final String DEFAULT_USERINFO = "Unknown User"; //$NON-NLS-1$

    /**
     * The at_hash claim name.
     */
    protected static final String AT_HASH_CLAIM = "at_hash"; //$NON-NLS-1$

    private static final String APPID_CLAIM = "appid"; //$NON-NLS-1$

    private static final String AZP_CLAIM = "azp"; //$NON-NLS-1$

    private static final String KID_CLAIM = "kid"; //$NON-NLS-1$

    private static final String ACR_CLAIM = "acr"; //$NON-NLS-1$

    private static final String COULD_NOT_VERIFY_OPEN_ID_CONNECT_AUTHENTICATION_REQUEST_ANSWER = "Could not verify OpenID Connect authentication request answer."; //$NON-NLS-1$

    private static final String MISSING_REQUIRED_ACCESS_TOKEN_FOR_THIS_FLOW = "Access Token is missing but required for this flow."; //$NON-NLS-1$

    private static final String MISSING_REQUIRED_ID_TOKEN = "ID Token is missing but required for this flow."; //$NON-NLS-1$

    private Map<String, ? extends Key> jwksKeyMap;

    /**
     * The Template Method. Consolidates logic from Client, Implicit, and Server AuthCode flows.
     */
    public JWTCheckResult check(Optional<String> rawIdToken, Optional<String> rawAccessToken) {
        Jwt<? extends Header, Claims> idTokenJWT = null;
        Jwt<? extends Header, Claims> accessTokenJWT = null;
        String userInfoEndpointResponse = null;
        String userInfo = DEFAULT_USERINFO; // Ensure this constant is accessible or passed in

        try {
            // First step: validate and parse provided id_token
            if (!rawIdToken.isPresent()) {
                if (isIdTokenMandatory()) {
                    throw new IllegalArgumentException(MISSING_REQUIRED_ID_TOKEN);
                }
            } else {
                idTokenJWT = validateAndParseIdToken(rawIdToken.get());
            }

            // Second step: validate and parse the optional access_token.
            if (!rawAccessToken.isPresent()) {
                if (isAccessTokenMandatory()) {
                    throw new IllegalArgumentException(MISSING_REQUIRED_ACCESS_TOKEN_FOR_THIS_FLOW);
                }
            } else {
                // We can only validate at_hash if we have an ID Token to compare against
                if (idTokenJWT != null) {
                    validateAtHash(rawAccessToken.get(), idTokenJWT);
                }
                accessTokenJWT = validateAndParseAccessToken(rawAccessToken.get());
            }

            // Third step: if required, call userInfoEndpoint and get optional userInfo response
            // We only call if logic dictates AND we actually have an access token to use
            if (rawAccessToken.isPresent() && shouldFetchUserInfo()) {
                userInfoEndpointResponse = callUserInfoEndpoint(rawAccessToken.get());
            }

            // Fourth step: retrieve user info
            userInfo = getUserInfo(idTokenJWT, accessTokenJWT, userInfoEndpointResponse);

            // Fifth step: post-validation authorization (hook)
            boolean accessGranted = postValidateAuthorization(idTokenJWT, accessTokenJWT, userInfoEndpointResponse);

            return new JWTCheckResult(userInfo, accessGranted);

        } catch (IllegalArgumentException | UnsupportedOperationException | JwtException e) {
            // Using a logger getter or static reference as per your snippet context
            logError(COULD_NOT_VERIFY_OPEN_ID_CONNECT_AUTHENTICATION_REQUEST_ANSWER, e);
            return new JWTCheckResult(userInfo, e.getMessage(), JWTCheckResult.SC_FORBIDDEN);
        }
    }

    /**
     * Retrieves and caches the JSON Web Key Set (JWKS) from the OpenID Connect provider. <br/>
     * This method fetches the public keys required to verify the digital signatures of incoming JWTs (such as ID
     * Tokens). The keys are retrieved from the endpoint defined by {@link #getJwksURI()}. <br/>
     * The keys are fetched only once.
     *
     * @return a {@link Map} mapping the Key ID ({@code kid}) to its corresponding Java {@link Key} object.
     * @throws SecurityException
     *             if the JWKS endpoint is unreachable, the HTTP request fails, or the response is empty.
     */
    protected Map<String, ? extends Key> getJwks() {
        if (jwksKeyMap == null) {
            String jwksURL = getJwksURI();
            StringBuilder responseStr = new StringBuilder();
            if (!HttpRequestHelper.callHttpService(jwksURL, new LinkedHashMap<>(), responseStr)) {
                throw new SecurityException("Error while trying to get JWKS from " + jwksURL); //$NON-NLS-1$
            }

            String jwks = responseStr.toString();
            if (StringUtil.isEmpty(jwks)) {
                throw new SecurityException("Empty JWKS received from " + jwksURL); //$NON-NLS-1$
            }

            jwksKeyMap = Jwks.setParser().build().parse(jwks).getKeys().stream().collect(Collectors.toMap(Identifiable::getId, Jwk::toKey));
        }
        return jwksKeyMap;
    }

    /**
     * Decodes, parses, and validates the ID Token (id_token) received from the OpenID Provider.
     * <p>
     * This method performs the following checks as per the OpenID Connect Core 1.0 specification:
     * <ul>
     * <li><b>Signature Validation:</b> Verifies the JWT signature using the public keys retrieved via
     * {@link #getJwks()}.</li>
     * <li><b>Issuer Check:</b> Ensures the {@code iss} claim matches the expected issuer URI
     * ({@link #getIssuerURI()}).</li>
     * <li><b>Audience Check:</b> Ensures the {@code aud} claim contains this client's Client ID
     * ({@link #getClientID()}).</li>
     * <li><b>Authorized Party (Optional):</b> If the {@code azp} claim is present, verifies it matches the Client
     * ID.</li>
     * <li><b>Application ID (Optional):</b> If the {@code appid} claim is present, verifies it matches the Client
     * ID.</li>
     * <li><b>Nonce Check:</b> Verifies that the {@code nonce} claim matches the one sent in the authorization request
     * (handled via {@link #checkNonce(Header, Claims)}).</li>
     * <li><b>Authentication Context Class Reference Check:</b> Verifies that the {@code acr} claim matches the
     * {@code acr_values} sent in the authorization request.</li>
     * </ul>
     * </p>
     *
     * @param idToken
     *            the raw, Base64URL-encoded ID Token string.
     * @return the parsed {@link Jwt} containing the header and claims.
     * @throws JwtException
     *             if the token is malformed, expired, has an invalid signature, or fails any claim validation check.
     * @throws IllegalArgumentException
     *             if the provided {@code idToken} is null or empty.
     */
    protected Jws<Claims> validateAndParseIdToken(String rawIdToken) {
        Map<String, ? extends Key> keyMap = getJwks();
        JwtParserBuilder jwtParserBuilder = Jwts.parser() //
                .keyLocator((Header h) -> keyMap.get(h.getOrDefault(KID_CLAIM, "").toString())) //$NON-NLS-1$
                .requireAudience(getClientID()) //
                .requireIssuer(getIssuerURI()) //
                .clockSkewSeconds(30);

        JwtParser jwtParser = jwtParserBuilder.build();
        Jws<Claims> jwt = jwtParser.parseSignedClaims(rawIdToken);

        JwsHeader header = jwt.getHeader();
        Claims claims = jwt.getPayload();

        checkNonce(header, claims);

        if (claims.containsKey(AZP_CLAIM) && !getClientID().equals(claims.get(AZP_CLAIM))) {
            throw new IncorrectClaimException(header, claims, AZP_CLAIM, claims.get(AZP_CLAIM), "Invalid value for \"azp\" claim in IdToken"); //$NON-NLS-1$
        }

        if (claims.containsKey(APPID_CLAIM) && !getClientID().equals(claims.get(APPID_CLAIM))) {
            throw new IncorrectClaimException(header, claims, APPID_CLAIM, claims.get(APPID_CLAIM), "Invalid value for \"appid\" claim in IdToken"); //$NON-NLS-1$
        }

        String acrValues = getAcrValues();
        if (!StringUtil.isEmpty(acrValues) && !(claims.get(ACR_CLAIM) instanceof String s && acrValues.contains(s))) {
            throw new IncorrectClaimException(header, claims, ACR_CLAIM, claims.get(ACR_CLAIM), "Invalid value for \"acr\" claim in IdToken - Authentication method might not be the expected one."); //$NON-NLS-1$
        }

        return jwt;
    }

    /**
     * Validates the Access Token Hash (at_hash) claim within the ID Token.
     * <p>
     * This check binds the Access Token to the ID Token, ensuring that the Access Token was not substituted or tampered
     * with during transit. The validation follows the algorithm defined in OIDC Core 1.0, Section 3.1.3.6:
     * <ol>
     * <li>Hash the {@code rawAccessToken} using the algorithm specified in the ID Token's header (e.g., SHA-256 for
     * RS256).</li>
     * <li>Take the left-most half of the hash.</li>
     * <li>Base64URL encode it.</li>
     * <li>Compare the result with the {@code at_hash} claim in the {@code idTokenJWT}.</li>
     * </ol>
     * </p>
     *
     * @param rawAccessToken
     *            the raw Access Token string.
     * @param idTokenJWT
     *            the already parsed and validated ID Token containing the {@code at_hash} claim.
     * @throws JwtException
     *             if the {@code at_hash} calculation fails or does not match the claim in the ID Token.
     * @throws UnsupportedOperationException
     *             if the ID Token's signing algorithm is not supported for hashing.
     */
    protected void validateAtHash(String rawAccessToken, Jwt<? extends Header, Claims> idToken) {
        // validate at_hash : if present in id_token
        String idTokenAlg = idToken.getHeader().getAlgorithm();
        String idTokenAtHash = (String) idToken.getPayload().get(AT_HASH_CLAIM); // $NON-NLS-1$

        if (StringUtil.isEmpty(idTokenAtHash)) {
            handleEmptyIdTokenAtHash(idToken);
            // Optional for Authentication Code Flow
            // Method can be overridden if so
        } else {
            verifyAtHash(rawAccessToken, idTokenAlg, idTokenAtHash);
        }
    }

    private MessageDigest getMessageDigest(String idTokenAlg) {
        MessageDigest md;
        Collection<String> sha256 = Stream.of(Jwts.SIG.ES256.getId(), Jwts.SIG.HS256.getId(), Jwts.SIG.RS256.getId(), Jwts.SIG.PS256.getId()).collect(Collectors.toCollection(HashSet::new));
        Collection<String> sha384 = Stream.of(Jwts.SIG.ES384.getId(), Jwts.SIG.HS384.getId(), Jwts.SIG.RS384.getId(), Jwts.SIG.PS384.getId()).collect(Collectors.toCollection(HashSet::new));
        Collection<String> sha512 = Stream.of(Jwts.SIG.ES512.getId(), Jwts.SIG.HS512.getId(), Jwts.SIG.RS512.getId(), Jwts.SIG.PS512.getId()).collect(Collectors.toCollection(HashSet::new));

        try {
            if (sha256.contains(idTokenAlg)) {
                md = MessageDigest.getInstance("SHA-256"); //$NON-NLS-1$
            } else if (sha384.contains(idTokenAlg)) {
                md = MessageDigest.getInstance("SHA-384"); //$NON-NLS-1$
            } else if (sha512.contains(idTokenAlg)) {
                md = MessageDigest.getInstance("SHA-512"); //$NON-NLS-1$
            } else {
                throw new SecurityException("Access token cannot be verified: unsupported alg value in IdToken header: " + idTokenAlg); //$NON-NLS-1$
            }
        } catch (NoSuchAlgorithmException e) {
            throw new SecurityException("Access token cannot be verified: no MessageDigest found for " + idTokenAlg, e); //$NON-NLS-1$
        }
        return md;
    }

    /**
     * Handles the "Math" pure logic. Can be called from tests, not intended to be called from other subclasses.
     * 
     * @param rawAccessToken
     * @param idTokenAlg
     * @param idTokenAtHash
     */
    protected void verifyAtHash(String rawAccessToken, String idTokenAlg, String idTokenAtHash) {
        MessageDigest md = getMessageDigest(idTokenAlg);
        byte[] asciiValue = rawAccessToken.getBytes(StandardCharsets.US_ASCII);
        byte[] encodedHash = md.digest(asciiValue);
        byte[] halfOfEncodedHash = Arrays.copyOf(encodedHash, encodedHash.length / 2);
        String computedAccessTokenHash = Base64.getUrlEncoder().withoutPadding().encodeToString(halfOfEncodedHash);
        if (!secureEquals(computedAccessTokenHash, idTokenAtHash)) {
            throw new SignatureException("Access token is not conform to provided IdToken alg and at_hash claims."); //$NON-NLS-1$
        }
    }

    /**
     * Securely compares two strings (e.g. hashes) to prevent timing attacks. Uses UTF-8 to enforce platform
     * independence.
     */
    protected boolean secureEquals(String a, String b) {
        if (a == null || b == null) {
            return false;
        }
        return MessageDigest.isEqual(a.getBytes(StandardCharsets.UTF_8), b.getBytes(StandardCharsets.UTF_8));
    }

    /**
     * Decodes, parses, and validates the Access Token (access_token), if required by configuration.
     * <p>
     * This method first checks {@link #needsAccessTokenJWTValidation()} to determine if validation should proceed. If
     * validation is required, it parses the token as a JWT and performs the following checks:
     * <ul>
     * <li><b>Signature Validation:</b> Verifies the JWT signature using the JWKS.</li>
     * <li><b>Issuer Check:</b> Ensures the {@code iss} claim matches the expected issuer.</li>
     * <li><b>Audience Check:</b> Optionally verifies the {@code aud} claim against {@link #getAccessTokenAudience()}
     * (if configured).</li>
     * <li><b>Authorized Party Check:</b> If the {@code azp} claim is present, ensures it matches the Client ID.</li>
     * </ul>
     * </p>
     * <p>
     * <b>Note on Opaque Tokens:</b> If {@link #needsAccessTokenJWTValidation()} returns {@code false} (e.g., because
     * the token is opaque and intended for the UserInfo endpoint), this method will simply return {@code null} without
     * throwing an exception.
     * </p>
     *
     * @param accessToken
     *            the raw Access Token string.
     * @return the parsed {@link Jwt} if validation succeeded, or {@code null} if validation was skipped.
     * @throws JwtException
     *             if validation is required but the token is malformed (e.g., opaque when JWT expected) or fails
     *             signature/claim checks.
     */
    protected Jwt<? extends Header, Claims> validateAndParseAccessToken(String accessToken) {
        if (!needsAccessTokenJWTValidation()) {
            // Do not validate token
            return null;
        }

        // If JWT token validation is required, we need to be able to check the signature.
        // For Entra ID:
        // - default access token are intended to be used by the Entra/Azure Graph API. The signature cannot be
        // validated not the issuer and the aud.
        // - custom scope / api exposed in app registration and v2 tokens are needed to be able to receive and validate
        // the JWT access_token.

        Map<String, ? extends Key> keyMap = getJwks();
        JwtParserBuilder jwtParserBuilder = Jwts.parser() //
                .keyLocator((Header h) -> keyMap.get(h.getOrDefault(KID_CLAIM, "").toString())) //$NON-NLS-1$
                .requireIssuer(getIssuerURI()) //
                .clockSkewSeconds(30);

        String accessTokenAudience = getAccessTokenAudience();
        if (!StringUtil.isEmpty(accessTokenAudience)) {
            jwtParserBuilder.requireAudience(accessTokenAudience);
        }

        JwtParser jwtParser = jwtParserBuilder.build();
        Jws<Claims> jwt = jwtParser.parseSignedClaims(accessToken);

        JwsHeader header = jwt.getHeader();
        Claims claims = jwt.getPayload();

        if (claims.containsKey(AZP_CLAIM) && !getClientID().equals(claims.get(AZP_CLAIM))) {
            throw new IncorrectClaimException(header, claims, AZP_CLAIM, claims.get(AZP_CLAIM), "Invalid value for \"azp\" claim in AccessToken"); //$NON-NLS-1$
        }

        return jwt;
    }

    private String callUserInfoEndpoint(String rawAccessToken) {
        String userInfoEndpointPayload = null;

        // Call the API to get the user information
        StringBuilder responseStr = new StringBuilder();
        Map<String, String> requestProperties = new LinkedHashMap<>();
        requestProperties.put("Authorization", "Bearer " + rawAccessToken); //$NON-NLS-1$//$NON-NLS-2$
        responseStr = new StringBuilder();
        if (HttpRequestHelper.callHttpService(getUserInfoEndPointURI(), requestProperties, responseStr)) {
            userInfoEndpointPayload = responseStr.toString();
        } else {
            throw new SecurityException("Error while calling /userInfo endpoint " + getUserInfoEndPointURI()); //$NON-NLS-1$
        }
        return userInfoEndpointPayload;
    }

    /**
     * Retrieve the user info in the passed objects regarding the current helper configuration.
     * 
     * @param idToken
     * @param accessToken
     * @param userInfoEndpointResponse
     * @return
     */
    protected String getUserInfo(Jwt<? extends Header, Claims> idToken, Jwt<? extends Header, Claims> accessToken, String userInfoEndpointResponse) {
        String userInfoEndPayload = getUserInfoPayload();
        String userInfoMatchClaim = getUserInfoMatchClaim();
        String userInfo = "Unknown User"; //$NON-NLS-1$ ;
        if (OpenIdConnectConstants.ID_TOKEN.equals(userInfoEndPayload)) {
            if (idToken != null) {
                Claims claims = idToken.getPayload();
                if (claims.containsKey(userInfoMatchClaim)) {
                    userInfo = (String) claims.get(userInfoMatchClaim);
                } else {
                    throw new IllegalArgumentException(String.format("Claim \"%1$s\" not found in IdToken", userInfoMatchClaim)); //$NON-NLS-1$
                }
            } else {
                throw new IllegalArgumentException(String.format("No IdToken provided to look for \"%1$s\" claim", userInfoMatchClaim)); //$NON-NLS-1$
            }
        } else if (OpenIdConnectConstants.ACCESS_TOKEN.equals(userInfoEndPayload)) {
            if (accessToken != null) {
                Claims claims = accessToken.getPayload();
                if (claims.containsKey(userInfoMatchClaim)) {
                    userInfo = (String) claims.get(userInfoMatchClaim);
                } else {
                    throw new IllegalArgumentException(String.format("Claim \"%1$s\" not found in AccessToken", userInfoMatchClaim)); //$NON-NLS-1$
                }
            } else {
                throw new IllegalArgumentException(String.format("No AccessToken provided to look for \"%1$s\" claim", userInfoMatchClaim)); //$NON-NLS-1$
            }
        } else {
            if (!StringUtil.isEmpty(userInfoEndpointResponse)) {
                Map<?, ?> claims = new Gson().fromJson(userInfoEndpointResponse, Map.class);
                if (claims.containsKey(userInfoMatchClaim)) {
                    userInfo = (String) claims.get(userInfoMatchClaim);
                } else {
                    // invalid value
                    throw new IllegalArgumentException(String.format("Claim \"%1$s\" not found in the JSON response: %2$s", userInfoMatchClaim, userInfoEndpointResponse)); //$NON-NLS-1$
                }
            } else {
                throw new IllegalArgumentException(String.format("No /userInfo response provided to look for \"%1$s\" claim", userInfoMatchClaim)); //$NON-NLS-1$
            }
        }

        return userInfo;
    }

    /**
     * Does this flow require an ID Token to proceed? Default: true (Most flows require it)
     */
    protected boolean isIdTokenMandatory() {
        return true;
    }

    /**
     * .
     */
    protected abstract void checkNonce(Header header, Claims claims);

    /**
     * Does this flow require an Access Token to proceed? Default: true
     */
    protected boolean isAccessTokenMandatory() {
        return true;
    }

    /**
     * Defines the behavior when the {@code at_hash} claim is missing from the ID Token. <br/>
     * This hook is called internally by {@link #validateAtHash(String, Jwt)} if the provided ID Token does not contain
     * an {@code at_hash} claim. <br/>
     * For the <b>Authorization Code Flow:</b> If the ID Token is returned from the Token Endpoint, the {@code at_hash}
     * claim is <b>OPTIONAL</b>. Default implementation returns nothing.
     *
     * @param idToken
     *            the parsed ID Token which was found to be missing the {@code at_hash} claim.
     * @throws io.jsonwebtoken.JwtException
     *             if the {@code at_hash} claim is mandatory for the current flow configuration.
     */
    protected void handleEmptyIdTokenAtHash(Jwt<? extends Header, Claims> idToken) {
        // Nothing to do : the at_hash claim is optional for Authentication Code Flow
    }

    /**
     * Determines whether the Access Token must be validated as a JWT signed by the issuer.
     * <p>
     * <b>Default:</b> {@code true}.
     * </p>
     * <p>
     * <b>When to override:</b> Return {@code false} if you are using Opaque Access Tokens (common when the token is
     * only intended for the UserInfo Endpoint) or if you want to rely solely on the Identity Provider's response from
     * the UserInfo endpoint validation instead of local signature verification.
     * </p>
     *
     * @return {@code true} to enforce JWT parsing and signature validation; {@code false} to skip it.
     */
    protected boolean needsAccessTokenJWTValidation() {
        return true;
    }

    /**
     * Determine if we should call the User Info Endpoint. Implementation varies between Client (checks UserInfoPayload)
     * and Server (checks RepositoryPayload).
     */
    protected abstract boolean shouldFetchUserInfo();

    /**
     * Hook to perform final business logic checks after tokens are validated. Default: returns true (access granted if
     * tokens are valid).
     */
    protected boolean postValidateAuthorization(Jwt<? extends Header, Claims> idToken, Jwt<? extends Header, Claims> accessToken, String userInfoResponse) {
        return true;
    }

    /**
     * Retrieves the OIDC {@code response_type} parameter used in the authorization request.
     * <p>
     * This value dictates which tokens are returned by the Identity Provider (e.g., "code" for Authorization Code Flow,
     * "id_token token" for Implicit Flow). It is also used internally to determine which tokens are mandatory for
     * validation (e.g., if it contains "id_token", the ID Token is required).
     * </p>
     *
     * @return the space-separated response type string (e.g., "code", "id_token token").
     */
    protected abstract String getResponseType();

    /**
     * Retrieves the expected Issuer URI ({@code iss}) of the OpenID Provider.
     * <p>
     * This value is strictly matched against the {@code iss} claim in both the ID Token and the Access Token (if
     * validated) to prevent token substitution attacks.
     * </p>
     *
     * @return the Issuer URI as a string (e.g., "https://login.microsoftonline.com/{tenant-id}/v2.0").
     */
    protected abstract String getIssuerURI();

    /**
     * Retrieves the Client ID ({@code client_id}) assigned to this application by the Identity Provider.
     * <p>
     * This value is used to:
     * <ul>
     * <li>Identify the application in authorization requests.</li>
     * <li>Validate the {@code aud} (Audience) claim in the ID Token.</li>
     * <li>Validate the {@code azp} (Authorized Party) claim in both tokens.</li>
     * </ul>
     * </p>
     *
     * @return the Client ID string.
     */
    protected abstract String getClientID();

    /**
     * Retrieves the URI of the JSON Web Key Set (JWKS) endpoint.
     * <p>
     * The helper uses this URI to fetch the public keys required to verify the digital signatures of incoming JWTs.
     * </p>
     *
     * @return the JWKS URI string.
     */
    protected abstract String getJwksURI();

    /**
     * Retrieves the URI of the UserInfo Endpoint.
     * <p>
     * If configured, the helper may call this endpoint using the Access Token to retrieve user attributes. This is
     * often used as an alternative to extracting claims directly from tokens or to validate Opaque Access Tokens.
     * </p>
     *
     * @return the UserInfo Endpoint URI, or {@code null} if not used.
     */
    protected abstract String getUserInfoEndPointURI();

    /**
     * Retrieves the name of the claim used to extract the User Identifier.
     * <p>
     * This claim name (e.g., "sub", "email", "oid", "upn") is used to extract the user's identity from:
     * <ol>
     * <li>The UserInfo Endpoint response (if called).</li>
     * <li>The Access Token (if it is a JWT and UserInfo was not used).</li>
     * <li>The ID Token (as a fallback).</li>
     * </ol>
     * </p>
     *
     * @return the claim key string.
     */
    protected abstract String getUserInfoMatchClaim();

    /**
     * Retrieves the configuration payload determining the source of the User Info.
     * <p>
     * This value helps the helper decide whether to call the UserInfo endpoint. Typically, if this equals
     * {@link org.eclipse.net4j.util.security.OpenIdConnectConstants#USER_INFO_ENDPOINT}, the helper will fetch data
     * from the remote endpoint; otherwise, it may expect data to be present in the tokens.
     * </p>
     *
     * @return the configuration string (URI or keyword) indicating the user info source.
     */
    protected abstract String getUserInfoPayload();

    /**
     * Retrieves the expected Audience ({@code aud}) for the Access Token.
     * <p>
     * If returned, the Access Token's {@code aud} claim must contain this value. This is useful when your server is the
     * intended resource server for the token. If {@code null} or empty, the Access Token audience check is skipped.
     * </p>
     *
     * @return the expected audience string, or {@code null} to skip the check.
     */
    protected abstract String getAccessTokenAudience();

    /**
     * Retrieves the optional "Authentication Context Class Reference" ({@code acr}) values.
     * <p>
     * If provided, these values are sent in the authorization request to specify the required authentication strength
     * (e.g., "mfa", "phr"). The helper will then validate that the {@code acr} claim in the received token matches one
     * of these values.
     * </p>
     *
     * @return a space-separated string of required ACR values, or {@code null} if no specific level is required.
     */
    protected abstract String getAcrValues();

    /**
     * 
     * Abstract method to handle logging by subclasses.
     * 
     * @param msg
     * @param e
     */
    protected abstract void logError(String msg, Exception e);

}
