/*******************************************************************************
 * Copyright (c) 2021, 2025 THALES GLOBAL SERVICES
 * All rights reserved.
 *******************************************************************************/
package fr.obeo.dsl.viewpoint.collab.common.user.protocol;

import java.util.Optional;

/**
 * A POJO class to hold information returned for ICollabProtocolConstants.SIGNAL_ID_GET_OPENID_AUTHENTIFICATION_INFO.
 * 
 * @author <a href="mailto:laurent.fasani@obeo.fr">Laurent Fasani</a>
 *
 */
public final class OpenIDAuthenticationInfo {

    private final boolean enabled;

    private OIDCConfig config;

    /**
     * Private constructor.
     */
    private OpenIDAuthenticationInfo(boolean enabled, OIDCConfig config) {
        this.enabled = enabled;
        this.config = config;
    }

    /**
     * Static factory for OpenID Connect disabled cases.
     */
    public static OpenIDAuthenticationInfo disabled() {
        return new OpenIDAuthenticationInfo(false, null);
    }

    /**
     * Static factory for OpenID Connect enabled cases.
     * 
     * @param config
     *            an OIDC configuration.
     */
    public static OpenIDAuthenticationInfo enabled(OIDCConfig config) {
        return new OpenIDAuthenticationInfo(true, config);
    }

    public boolean isEnabled() {
        return enabled;
    }

    public Optional<OIDCConfig> getConfig() {
        return Optional.ofNullable(config);
    }

    public boolean isAuthenticationFlowWithPKCE() {
        return enabled && config instanceof AuthenticationCodeFlowWithPKCE;
    }

    public boolean isLegacyImplicitFlow() {
        return enabled && config instanceof LegacyImplicitFlow;
    }

    // Only permits specific implementations, preventing unauthorized extensions
    public sealed interface OIDCConfig permits LegacyImplicitFlow, AuthenticationCodeFlowWithPKCE {
    }

    /**
     * Configuration for the OpenID Connect Implicit Flow.
     * <p>
     * This record holds all necessary info to trigger the client-side login.
     * </p>
     */
    public record LegacyImplicitFlow(boolean httpsEnabled, int port) implements OIDCConfig {
    }

    /**
     * Configuration for the OpenID Connect Authorization Code Flow with PKCE.
     * <p>
     * This record holds all necessary endpoints and identifiers required to perform a secure login using Proof Key for
     * Code Exchange (PKCE).
     * </p>
     */
    public record AuthenticationCodeFlowWithPKCE(//
            String responseType, //
            String clientID, //
            String scope, //
            String acrValues, //
            String complement, //
            String authorizationEndpoint, //
            String tokenEndpoint, //
            String jwksUri, //
            String issuer, //
            String userInfoEndpoint, //
            String userInfoPayload, //
            String userInfoMatchClaim, //
            int loginTimeoutMS, //
            String loopbackServerName, //
            int loopbackServerPort //
    ) implements OIDCConfig {

        /**
         * Creates a new Builder instance for constructing the configuration.
         * 
         * @return a new {@link Builder}
         */
        public static Builder builder() {
            return new Builder();
        }

        /**
         * Builder for {@link AuthenticationCodeFlowWithPKCE}.
         */
        public static class Builder {
            private String responseType;

            private String clientID;

            private String scope;

            private String acrValues;

            private String complement;

            private String authorizationEndpoint;

            private String tokenEndpoint;

            private String jwksUri;

            private String issuer;

            private String userInfoEndpoint;

            private String userInfoPayload;

            private String userInfoMatchClaim;

            private int loginTimeoutMS;

            private String loopbackServerName;

            private int loopbackServerPort;

            /**
             * Sets the Response Type.
             * <p>
             * Defaults to {@code "code"}. For PKCE, this should almost always be "code".
             * </p>
             * * @param resp the OAuth2 response type
             * 
             * @return this builder
             */
            public Builder responseType(String resp) {
                this.responseType = resp;
                return this;
            }

            /**
             * Sets the Client ID.
             * <p>
             * This is the public identifier for the application as registered with the Identity Provider (IdP).
             * </p>
             * * @param id the client identifier (e.g., "my-eclipse-app")
             * 
             * @return this builder
             */
            public Builder clientID(String id) {
                this.clientID = id;
                return this;
            }

            /**
             * Sets the requested Scopes.
             * <p>
             * A space-separated list of permissions requested from the user. Must typically include "openid" for
             * authentication.
             * </p>
             * * @param s the scope string (e.g., "openid profile email api://my-api/access")
             * 
             * @return this builder
             */
            public Builder scope(String s) {
                this.scope = s;
                return this;
            }

            /**
             * Sets the ACR Values for the authentication request, will then be check as present in the ID Token acr claim.
             * <p>
             * * @param acr the acr values (a+b+c)
             * 
             * @return this builder
             */
            public Builder acrValues(String acr) {
                this.acrValues = acr;
                return this;
            }

            /**
             * Sets supplementary configuration information.
             * <p>
             * Can be used for extra context, query parameters, or legacy compatibility strings.
             * </p>
             * * @param cplt the complement string
             * 
             * @return this builder
             */
            public Builder complement(String cplt) {
                this.complement = cplt;
                return this;
            }

            /**
             * Sets the Authorization Endpoint URL.
             * <p>
             * The URL where the user's browser is redirected to start the login process.
             * </p>
             * * @param endpoint the full URL of the authorization endpoint
             * 
             * @return this builder
             */
            public Builder authorizationEndpoint(String endpoint) {
                this.authorizationEndpoint = endpoint;
                return this;
            }

            /**
             * Sets the Token Endpoint URL.
             * <p>
             * The URL used by the application to exchange the Authorization Code (and PKCE verifier) for the ID Token
             * and Access Token.
             * </p>
             * * @param endpoint the full URL of the token endpoint
             * 
             * @return this builder
             */
            public Builder tokenEndpoint(String endpoint) {
                this.tokenEndpoint = endpoint;
                return this;
            }

            /**
             * Sets the Issuer Identifier.
             * <p>
             * The unique identifier of the Identity Provider (e.g., "https://login.microsoftonline.com/{tenant}/v2.0").
             * This must match the {@code iss} claim in the received tokens exactly.
             * </p>
             * * @param iss the issuer string
             * 
             * @return this builder
             */
            public Builder issuer(String iss) {
                this.issuer = iss;
                return this;
            }

            /**
             * Sets the JWKS (JSON Web Key Set) URI.
             * <p>
             * The URL where the Identity Provider publishes its public keys. Used to validate the signature of the ID
             * Token and Access Token.
             * </p>
             * * @param endpoint the full URL of the keys endpoint (e.g., ".../discovery/v2.0/keys")
             * 
             * @return this builder
             */
            public Builder jwksUri(String endpoint) {
                this.jwksUri = endpoint;
                return this;
            }

            /**
             * Sets the User Info Endpoint URL.
             * <p>
             * The URL where the Identity Provider provides user info.
             * </p>
             * * @param endpoint the full URL of the user info endpoint
             * 
             * @return this builder
             */
            public Builder userInfoEndpoint(String endpoint) {
                this.userInfoEndpoint = endpoint;
                return this;
            }

            /**
             * Sets the user info payload: id_token | access_token | userInfoEndpoint. Default: userInfoEndpoint.
             * <p>
             * The payload in which the user info to display will be looked for.
             * </p>
             * * @param payload id_token | access_token | userInfoEndpoint
             * 
             * @return this builder
             */
            public Builder userInfoPayload(String payload) {
                this.userInfoPayload = payload;
                return this;
            }

            /**
             * Sets the user info match claim.
             * <p>
             * The claim name that will be used to get the user info to display
             * </p>
             * * @param claim a claim attribute name
             * 
             * @return this builder
             */
            public Builder userInfoMatchClaim(String claim) {
                this.userInfoMatchClaim = claim;
                return this;
            }

            /**
             * Sets the Login Timeout.
             * <p>
             * The maximum time (in milliseconds) the local server will wait for the redirect callback from the browser
             * before aborting. Defaults to 60,000ms (1 minute).
             * </p>
             * * @param timeoutMS timeout in milliseconds
             * 
             * @return this builder
             */
            public Builder loginTimeoutMS(int timeoutMS) {
                this.loginTimeoutMS = timeoutMS;
                return this;
            }

            /**
             * Sets the hostname for the local loopback server.
             * <p>
             * Defaults to {@code "localhost"}. This is the interface the application binds to while waiting for the
             * OIDC callback.
             * </p>
             * * @param name the hostname (usually "localhost" or "127.0.0.1")
             * 
             * @return this builder
             */
            public Builder loopbackServerName(String name) {
                this.loopbackServerName = name;
                return this;
            }

            /**
             * Sets the specific port for the local loopback server.
             * <p>
             * Defaults to {@code 0}, which allows the system to pick a random available ephemeral port. Only set a
             * specific port if your Identity Provider requires an exact, pre-registered Redirect URI.
             * </p>
             * * @param port the port number, or 0 for random
             * 
             * @return this builder
             */
            public Builder loopbackServerPort(int port) {
                this.loopbackServerPort = port;
                return this;
            }

            /**
             * Builds the {@link AuthenticationCodeFlowWithPKCE} configuration instance. * @return the immutable
             * configuration record
             * 
             * @throws IllegalStateException
             *             if required fields (like ClientID) are missing
             */
            public AuthenticationCodeFlowWithPKCE build() {
                return new AuthenticationCodeFlowWithPKCE(responseType, //
                        clientID, //
                        scope, //
                        acrValues, //
                        complement, //
                        authorizationEndpoint, //
                        tokenEndpoint, //
                        jwksUri, //
                        issuer, //
                        userInfoEndpoint, //
                        userInfoPayload, //
                        userInfoMatchClaim, //
                        loginTimeoutMS, //
                        loopbackServerName, //
                        loopbackServerPort);
            }
        }
    }

}
