/*******************************************************************************
 * Copyright (c) 2025, 2026 THALES GLOBAL SERVICES.
 * All rights reserved.
 *
 * Contributors:
 *    Obeo - initial API and implementation
 *******************************************************************************/
package fr.obeo.dsl.viewpoint.collab.common.internal.jwt;

import java.util.Optional;

import org.eclipse.core.runtime.IStatus;

import fr.obeo.dsl.viewpoint.collab.common.internal.Activator;
import fr.obeo.dsl.viewpoint.collab.common.internal.openid.OpenIdConnectConstants;
import fr.obeo.dsl.viewpoint.collab.common.user.protocol.OpenIDAuthenticationInfo.AuthenticationCodeFlowWithPKCE;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Header;
import io.jsonwebtoken.IncorrectClaimException;
import io.jsonwebtoken.MissingClaimException;

/**
 * Helper to validate OpenID Connect received id_token and optional access_token for the server-side implementation.
 * 
 * This helper uses the JJWT library.
 * 
 * References:
 * <ul>
 * <li>JJWT: https://github.com/jwtk/jjwt</li>
 * <li>OpenID Connect Core 1.0: https://openid.net/specs/openid-connect-core-1_0.html
 * <li>Other OpenID Connect specs: https://openid.net/developers/specs/</li>
 * </ul>
 * 
 * @author mporhel
 */
public class AuthenticationCodeFlowClientJWTHelper extends AbstractJWTHelper {

    private AuthenticationCodeFlowWithPKCE oidcInfo;

    private Optional<String> expectedNonce;

    /**
     * Constructor.
     * 
     * @param oidcInfo
     *            its OpenID configuration
     */
    public AuthenticationCodeFlowClientJWTHelper(AuthenticationCodeFlowWithPKCE oidcInfo) {
        this(oidcInfo, Optional.empty());

    }

    /**
     * Constructor.
     * 
     * @param oidcInfo
     *            its OpenID configuration
     * @param expectedNonce
     *            the nonce to check if present
     */
    public AuthenticationCodeFlowClientJWTHelper(AuthenticationCodeFlowWithPKCE oidcInfo, Optional<String> expectedNonce) {
        this.expectedNonce = expectedNonce;
        this.oidcInfo = oidcInfo;

    }

    @Override
    protected boolean isIdTokenMandatory() {
        // In the Authentication Code Flow, we just exchanged a code and got the Token Response from the Authorization
        // Server.
        // It must contains an ID Token to verify the identity of who we just logged in.
        return true;
    }

    @Override
    protected void checkNonce(Header header, Claims claims) {
        // Nonce is optional for the authentication code and present on the first id_token traded against the code,
        // there is no new nonce or even no nonce after refresh token requests.
        if (expectedNonce.isPresent()) {
            if (!claims.containsKey(NONCE_CLAIM)) {
                throw new MissingClaimException(header, claims, NONCE_CLAIM, null, "No \"nonce\" claim in IdToken"); //$NON-NLS-1$
            } else if (!(claims.get(NONCE_CLAIM) instanceof String nonce && secureEquals(expectedNonce.get(), nonce))) {
                throw new IncorrectClaimException(header, claims, NONCE_CLAIM, claims.get(NONCE_CLAIM), "Invalid value for \"nonce\" claim  in IdToken"); //$NON-NLS-1$
            }
        }
    }

    @Override
    protected boolean isAccessTokenMandatory() {
        // In the Authentication Code Flow, we just exchanged a code and got the Token Response from the Authorization
        // Server.
        // It MUST contains an Access Token per the OAuth 2.0 / OIDC specifications.
        // It is required to call the UserInfo endpoint or to authorize subsequent API requests.
        return true;
    }

    @Override
    protected boolean needsAccessTokenJWTValidation() {
        return OpenIdConnectConstants.ACCESS_TOKEN.equals(getUserInfoPayload());
    }

    @Override
    protected boolean shouldFetchUserInfo() {
        return OpenIdConnectConstants.USER_INFO_ENDPOINT.equals(getUserInfoPayload());
    }

    @Override
    protected String getResponseType() {
        return oidcInfo.responseType();
    }

    @Override
    protected String getIssuerURI() {
        return oidcInfo.issuer();
    }

    @Override
    protected String getClientID() {
        return oidcInfo.clientID();
    }

    @Override
    protected String getJwksURI() {
        return oidcInfo.jwksUri();
    }

    @Override
    protected String getUserInfoEndPointURI() {
        return oidcInfo.userInfoEndpoint();
    }

    @Override
    protected String getUserInfoMatchClaim() {
        return oidcInfo.userInfoMatchClaim();
    }

    @Override
    protected String getUserInfoPayload() {
        return oidcInfo.userInfoPayload();
    }

    @Override
    protected String getAccessTokenAudience() {
        // No access token audience check on the client.
        return null;
    }

    @Override
    protected String getAcrValues() {
        return oidcInfo.acrValues();
    }

    @Override
    protected void logError(String msg, Exception e) {
        Activator.getDefault().log(IStatus.ERROR, msg, e);
    }
}
