/*******************************************************************************
 * Copyright (c) 2025 THALES GLOBAL SERVICES and others.
 * All rights reserved.
 *
 * Contributors:
 *    Obeo - initial API and implementation
 *******************************************************************************/
package fr.obeo.dsl.viewpoint.collab.common.internal.security;

import java.security.SecureRandom;
import java.util.UUID;

import org.bouncycastle.crypto.RuntimeCryptoException;
import org.bouncycastle.crypto.generators.OpenBSDBCrypt;

import fr.obeo.dsl.viewpoint.collab.common.internal.Activator;

/**
 * This class use BCrypt to generate and to check tokens.
 * 
 * @author lfasani
 *
 */
public final class BCryptService {

    private static final String VERSION = "2b"; //$NON-NLS-1$

    private static final int COST = 12;

    private BCryptService() {
    }

    /**
     * Generate a new token.
     * 
     * @return a new random token.
     */
    public static char[] generateRandomToken() {
        return UUID.randomUUID().toString().toCharArray();
    }

    /**
     * Generate a new encrypted token.
     * 
     * @return a 60 character Bcrypt String.
     */
    public static String hash(char[] password) {
        byte[] salt = new byte[16];
        new SecureRandom().nextBytes(salt);
        return OpenBSDBCrypt.generate(VERSION, password, salt, COST);
    }

    /**
     * Check if the password corresponds to the BCrypt password.
     * 
     * @param bcryptString
     *            a 60 character Bcrypt String, including version, cost factor, salt and hash, separated by '$'.
     * @param password
     *            the password to check.
     * @return true if the password is correct.
     */
    public static boolean verify(String bcryptString, char[] password) {
        boolean checkPassword = false;
        try {
            checkPassword = OpenBSDBCrypt.checkPassword(bcryptString, password);
        } catch (IllegalArgumentException | RuntimeCryptoException e) {
            Activator.getDefault().getLog().error("Impossible to verify credentials", e); //$NON-NLS-1$
        }
        return checkPassword;
    }

}
