/*******************************************************************************
 * Copyright (c) 2022, 2025 THALES GLOBAL SERVICES. All rights reserved.
 *
 * Contributors: Obeo - initial API and implementation
 *******************************************************************************/
package fr.obeo.dsl.viewpoint.collab.common.internal.security;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.KeySpec;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collection;
import java.util.Optional;
import java.util.UUID;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.PBEKeySpec;
import javax.crypto.spec.SecretKeySpec;

import org.eclipse.equinox.internal.security.storage.SecurePreferencesMapper;
import org.eclipse.equinox.internal.security.storage.SecurePreferencesWrapper;
import org.eclipse.equinox.security.storage.ISecurePreferences;
import org.eclipse.equinox.security.storage.SecurePreferencesFactory;
import org.eclipse.equinox.security.storage.StorageException;

import fr.obeo.dsl.viewpoint.collab.common.internal.Activator;

/**
 * A service class to read credentials from the secure storage to send http request.
 * 
 * @author lfasani
 *
 */
public class HttpClientSecureStorageService {

    private static final String PWD_ATT = "password"; //$NON-NLS-1$

    private static final String LOGIN_ATT = "login"; //$NON-NLS-1$

    private static final String SEPARATOR = ":"; //$NON-NLS-1$

    private static final String ENCRYPTION_ALGORYTHM = "AES/CBC/PKCS5Padding"; //$NON-NLS-1$

    private static final String ENCRYPTION_KEY_ALGORITHM = "AES"; //$NON-NLS-1$

    private static final String PBKDF_ALGORITHM = "PBKDF2WithHmacSHA256"; //$NON-NLS-1$

    private static final char[] CHAR_ARRAY = { 'U', 'n', 's', 'w', 'o', 'r', 'n', '6', 'F', 'a', 'l', 'c', 'o', 'n', '4', 'S', 'h' };

    private ISecurePreferences preferences;

    /**
     * Default constructor.
     */
    public HttpClientSecureStorageService() {
        this.preferences = SecurePreferencesFactory.getDefault();
    }

    /**
     * Default constructor.
     * 
     * @param preferences
     *            The {@link ISecurePreferences}. if null, the default on is used.
     * @param httpPort
     *            used to create the node in the secure storage that will be used by the application client
     */
    public HttpClientSecureStorageService(ISecurePreferences preferences) {
        if (preferences != null) {
            this.preferences = preferences;
        } else {
            this.preferences = SecurePreferencesFactory.getDefault();
        }
    }

    /**
     * Close the secure storage.
     */
    public void close() {
        SecurePreferencesMapper.close(((SecurePreferencesWrapper) this.preferences).getContainer().getRootData());
        SecurePreferencesMapper.clearDefault();
    }

    /**
     * Gets the credentials available in the secure storage.
     * 
     * @param hostName
     *            hostName of the admin server
     * @param httpPort
     *            port of the admin server
     * @param httpLogin
     *            the user for which the password is searched. If null, the couple user/password for the server is
     *            returned.
     */
    public HttpCredentials loadCredentialsFromSecureStorage(String httpHostName, int httpPort, String httpLogin) {
        ISecurePreferences credentialsNode = preferences.node(getToolsCredentialsSecureStorageKey());

        if (credentialsNode != null) {
            Optional<ISecurePreferences> serverNode = getNode(httpHostName, httpPort, credentialsNode);

            if (!serverNode.isPresent()) {
                // Fallback on first found entry for localhost
                Optional<ISecurePreferences> fallbackNode = getNode(SecureStorageService.DEFAULT_SERVER_HOST_NAME_FROM_CLIENT_APPLICATION, -1, credentialsNode);

                if (fallbackNode.isPresent()) {
                    Activator.getDefault().getLog().info("Falling back on secure storage node " + fallbackNode.get().name() + " to load http credentials."); //$NON-NLS-1$//$NON-NLS-2$
                    serverNode = fallbackNode;
                }
            }

            if (serverNode.isPresent()) {
                try {
                    String userAttr = serverNode.get().get(LOGIN_ATT, null);
                    boolean validUser = userAttr != null;
                    if (httpLogin != null && !httpLogin.isEmpty()) {
                        validUser = validUser && httpLogin.equals(userAttr);
                    }
                    if (validUser) {
                        String tokenAttr = serverNode.get().get(PWD_ATT, null);
                        Optional<String> decryptedToken = decrypt(tokenAttr);
                        if (decryptedToken.isPresent()) {
                            return new HttpCredentials(userAttr, decryptedToken.get().toCharArray());
                        }
                    }
                } catch (StorageException e) {
                    Activator.getDefault().getLog().error("Impossible to retrieve the user credentials from the secure storage for the server: " + httpHostName + SEPARATOR + httpPort, e); //$NON-NLS-1$
                }
            }
        }
        Activator.getDefault().getLog().error("Impossible to retrieve the user credentials from the secure storage for the server: " + httpHostName + SEPARATOR + httpPort); //$NON-NLS-1$
        return null;

    }

    private Optional<ISecurePreferences> getNode(String httpHostName, int httpPort, ISecurePreferences credentialsNode) {
        // @formatter:off
        Optional<ISecurePreferences> serverNode = Arrays.asList(credentialsNode.childrenNames()).stream()
                .filter(name -> httpPort == -1 ? name.startsWith(httpHostName + SEPARATOR) : name.equals(httpHostName + SEPARATOR + httpPort))
                .findFirst()
                .map(name-> credentialsNode.node(name));
        // @formatter:on

        return serverNode;
    }

    /**
     * Sets the credentials in the secure storage.
     * 
     * @param hostName
     *            hostName of the admin server
     * @param httpPort
     *            port of the admin server
     * @param httpLogin
     *            the user for which the password is searched. If null, the couple user/password for the server is
     *            returned.
     * @return true if the information has been properly saved in the secure storage.
     */
    public boolean setCredentialsIntoSecureStorage(String httpHostName, int httpPort, String httpLogin, char[] password) throws StorageException, IOException {
        ISecurePreferences credentialsNode = preferences.node(getToolsCredentialsSecureStorageKey());
        ISecurePreferences serverNode = credentialsNode.node(httpHostName + SEPARATOR + httpPort);

        Optional<String> encryptedPassword = this.encrypt(String.valueOf(password));
        if (encryptedPassword.isPresent()) {
            serverNode.put(LOGIN_ATT, httpLogin, false);
            serverNode.put(PWD_ATT, encryptedPassword.get(), false);
            credentialsNode.flush();
            return true;
        }
        return false;
    }

    /**
     * Return whether the secure-storage has been initialized with the tools http credentials.
     * 
     * @return true if the secure-storage contains the expected root node, false otherwise.
     */
    public boolean isHttpToolsCredentialsInitialized(String httpHostName, int httpPort) {
        boolean nodeExists = false;
        if (preferences.nodeExists(getToolsCredentialsSecureStorageKey())) {
            ISecurePreferences credentialsNode = preferences.node(getToolsCredentialsSecureStorageKey());
            String serverNodeName = httpHostName + SEPARATOR + httpPort;
            nodeExists = credentialsNode.nodeExists(serverNodeName);
        }
        return nodeExists;
    }

    /**
     * Clear the credentials in the secure storage.
     * 
     * @param hostName
     *            hostName of the admin server
     * @param httpPort
     *            port of the admin server
     * @return true if the credentials has been properly removed from the secure storage.
     */
    public boolean clearCredentialsIntoSecureStorage(String httpHostName, int httpPort) {
        if (preferences.nodeExists(getToolsCredentialsSecureStorageKey())) {
            ISecurePreferences credentialsNode = preferences.node(getToolsCredentialsSecureStorageKey());
            String serverNodeName = httpHostName + SEPARATOR + httpPort;
            if (credentialsNode.nodeExists(serverNodeName)) {
                credentialsNode.node(serverNodeName).removeNode();
                try {
                    preferences.flush();
                    return true;
                } catch (IOException e) {
                    Activator.getDefault().getLog().error("Impossible to clear the credentials.", e); //$NON-NLS-1$
                }
            }
        }
        return false;
    }

    /**
     * Clear the credentials fallbacks in the secure storage (entries with localhost as server name).
     * 
     * Also remove the tools http credentials root node if empty after fallback cleanup. Next server start will be able
     * to regen the initial admin token if this method has been call during Jetty realm removal.
     * 
     * @return true if the credentials have been properly removed from the secure storage.
     */
    public boolean clearSecureStorageCredentialsFallbacks() {
        if (preferences.nodeExists(getToolsCredentialsSecureStorageKey())) {
            boolean done = false;
            ISecurePreferences credentialsNode = preferences.node(getToolsCredentialsSecureStorageKey());

            Collection<ISecurePreferences> localhostNodes = Arrays.asList(credentialsNode.childrenNames()).stream() //
                    .filter(name -> name.startsWith(SecureStorageService.DEFAULT_SERVER_HOST_NAME_FROM_CLIENT_APPLICATION + SEPARATOR)) //
                    .map(name -> credentialsNode.node(name)) //
                    .toList();

            for (ISecurePreferences localhostNode : localhostNodes) {
                localhostNode.removeNode();
                done = true;
            }

            if (credentialsNode.childrenNames().length == 0) {
                credentialsNode.removeNode();
                done = true;
            }
            try {
                preferences.flush();
            } catch (IOException e) {
                Activator.getDefault().getLog().error("Impossible to clear the credential fallbacks.", e); //$NON-NLS-1$
            }
            return done;
        }
        return false;

    }

    private String getToolsCredentialsSecureStorageKey() {
        return SecureStorageKeys.HTTP_CREDENTIALS;
    }

    private Optional<String> encrypt(String input) {
        Optional<SecretKey> key = getCipherKey();
        if (key.isPresent()) {
            try {
                Cipher cipher = getCipher(Cipher.ENCRYPT_MODE, key.get());
                String salt = UUID.randomUUID().toString().substring(0, 16);
                byte[] cipherText = cipher.doFinal((salt + input).getBytes(StandardCharsets.UTF_8));
                String encodedString = Base64.getEncoder().encodeToString(cipherText);

                return Optional.of(encodedString);
            } catch (InvalidKeyException | InvalidAlgorithmParameterException | NoSuchAlgorithmException | NoSuchPaddingException | IllegalBlockSizeException | BadPaddingException e) {
                Activator.getDefault().getLog().error("Password encryption error."); //$NON-NLS-1$
            }
        }
        return Optional.empty();
    }

    private Optional<String> decrypt(String cipherText) {
        Optional<SecretKey> key = getCipherKey();
        if (key.isPresent()) {
            try {
                Cipher cipher = getCipher(Cipher.DECRYPT_MODE, key.get());
                byte[] plainText = cipher.doFinal(Base64.getDecoder().decode(cipherText));
                return Optional.of(new String(plainText).substring(16));
            } catch (InvalidKeyException | InvalidAlgorithmParameterException | NoSuchAlgorithmException | NoSuchPaddingException | IllegalBlockSizeException | BadPaddingException e) {
                Activator.getDefault().getLog().error("Password decryption error."); //$NON-NLS-1$
            }
        }
        return Optional.empty();
    }

    private Cipher getCipher(int opmode, SecretKey key) throws NoSuchAlgorithmException, NoSuchPaddingException, InvalidKeyException, InvalidAlgorithmParameterException {
        IvParameterSpec ivParameterSpec = this.generateInitializationVector();
        Cipher cipher = Cipher.getInstance(ENCRYPTION_ALGORYTHM);
        cipher.init(opmode, key, ivParameterSpec);
        return cipher;
    }

    private IvParameterSpec generateInitializationVector() {
        byte[] iv = "RU!^Mg29r45GJ5t!".getBytes(StandardCharsets.UTF_8); //$NON-NLS-1$
        return new IvParameterSpec(iv);
    }

    private Optional<SecretKey> getCipherKey() {
        Optional<SecretKey> secret = Optional.empty();
        SecretKeyFactory factory;
        try {
            factory = SecretKeyFactory.getInstance(PBKDF_ALGORITHM);
            KeySpec spec = new PBEKeySpec(CHAR_ARRAY, "277a94da".getBytes(StandardCharsets.UTF_8), 1000, 256); //$NON-NLS-1$
            secret = Optional.of(new SecretKeySpec(factory.generateSecret(spec).getEncoded(), ENCRYPTION_KEY_ALGORITHM));
        } catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
            Activator.getDefault().getLog().error("Password encryption/decryption initialization error."); //$NON-NLS-1$
        }
        return secret;
    }

}
