/*******************************************************************************
 * Copyright (c) 2017, 2024 THALES GLOBAL SERVICES
 * All rights reserved.
 *
 *  Contributors:
 *      Obeo - Initial API and implementation
 *******************************************************************************/
package fr.obeo.dsl.viewpoint.collab.common.internal.commit;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * A {@link InternalCommitDescriptor} aims to provides a way to summarize all information about a CDO commit.
 * 
 * @author <a href="mailto:arthur.daussy@obeo.fr">Arthur Daussy</a>
 *
 */
public final class InternalCommitDescriptor {

    /**
     * Property used to define a commit that is tagged a technical.
     */
    public static final String TN_COMMIT_PROPERTY = "technical-commit"; //$NON-NLS-1$

    /**
     * Pattern used to split lines in the description of the CDO commit
     */
    private static final String EOL_PATTERN = "\\r?\\n"; //$NON-NLS-1$

    private static final String EOL = System.lineSeparator();

    /**
     * Prefix used to ease the identification of the properties
     */
    private static final String PROP_PREFIX = "team-"; //$NON-NLS-1$

    /**
     * Pattern that should be used for the properties
     */
    private static final Pattern PROPERTY_PATTERN = Pattern.compile(PROP_PREFIX + "(.*?)\\s*:\\s*((.*?))?"); //$NON-NLS-1$

    /**
     * Maps of the properties of this commit
     */
    private final Map<String, String> properties;

    /**
     * Message of this commit
     */
    private final String message;

    /**
     * Simple constructor.
     * 
     * @param message
     *            the message of the commit
     * @param properties
     *            map of properties of this commit
     */
    public InternalCommitDescriptor(String message, Map<String, String> properties) {
        super();
        this.properties = new LinkedHashMap<>(properties);
        this.message = message;
    }

    /**
     * Creates a new {@link InternalCommitDescriptor} from a commit description.
     * 
     * @param comment
     *            a commit description
     * @return a new {@link InternalCommitDescriptor}
     */
    public static InternalCommitDescriptor createFromCommitComment(String comment) {

        if (comment == null) {
            return new InternalCommitDescriptor(null, Collections.emptyMap());
        }

        String[] lines = comment.split(EOL_PATTERN);

        StringBuilder messageBuilder = new StringBuilder();
        boolean propertyZone = false; // Will hold true when the parsing reach the property zone

        Map<String, String> properties = new LinkedHashMap<>(); // Keeps properties order

        for (int i = 0; i < lines.length; i++) {
            String line = lines[i];

            Matcher matcher = PROPERTY_PATTERN.matcher(line.trim());
            if (matcher.matches()) {
                // We are in the property zone
                propertyZone = true;
                String value = matcher.group(2);
                if (value != null && value.trim().isEmpty()) {
                    value = null;
                }
                properties.put(matcher.group(1), value);
            } else if (!propertyZone) { // Do not add lines that belong to the property zone
                messageBuilder.append(line).append(EOL);
            }
        }

        String message = messageBuilder.toString();
        return new InternalCommitDescriptor(message.isEmpty() ? null : message, properties);

    }

    /**
     * Gets the property with a given name.
     * 
     * @param propertyName
     *            the name of the property
     * @return the value or <code>null</code> if none
     */
    public String getProperty(String propertyName) {
        return properties.get(propertyName);
    }

    /**
     * Returns a view of all properties of this commit. The returned {@link Map} is not modifiable.
     * 
     * @return a view of all properties of this commit.
     */
    public Map<String, String> getAllProperties() {
        return Collections.unmodifiableMap(this.properties);
    }

    /**
     * Returns <code>true</code> if the current commit is a technical commit.
     * 
     * @return <code>true</code> if the current commit is a technical commit, <code>false</code> otherwise
     */
    public boolean isTechnicalCommit() {
        return Boolean.valueOf(getProperty(TN_COMMIT_PROPERTY));
    }

    /**
     * Gets the boolean value of a properties.
     * 
     * @param propertyName
     *            the name of the property
     * @return <code>true</code> if the value of the property is equals to "true", <code>false</code> in all other cases
     */
    public boolean getBooleanProperty(String propertyName) {
        String value = properties.get(propertyName);
        if (value != null) {
            return Boolean.valueOf(value);
        }
        return false;
    }

    /**
     * Gets the message of the commit.
     * 
     * @return a String (or <code>null</code> if there is no message)
     */
    public String getMessage() {
        return message;
    }

    @Override
    public String toString() {
        StringBuilder builder = new StringBuilder();

        if (message != null && !message.isEmpty()) {
            String[] lines = message.split(EOL_PATTERN);
            for (String line : lines) {
                builder.append(line).append(EOL);
            }
            builder.append(EOL);
        }

        for (Entry<String, String> prop : properties.entrySet()) {
            builder.append(PROP_PREFIX).append(prop.getKey()).append(" : "); //$NON-NLS-1$
            String value = prop.getValue();

            // Do not display null value
            if (value != null) {
                builder.append(value);
            }
            builder.append(EOL);

        }

        return builder.toString();
    }

}
